//==========================================================================
// EliMZ_ButtonCommonEvents.js
//==========================================================================

/*:
@target MZ
@base EliMZ_Book

@plugindesc v2.1.1 - Bind common events to keyboard/gamepad buttons!
@author Hakuen Studio
@url https://hakuenstudio.itch.io/

@help
▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬
If you like my work, please consider supporting me on Patreon!
https://www.patreon.com/hakuenstudio
▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬
============================================================================
Introduction
============================================================================

This plugin lets you bind common events to any keyboard button AND gamepad 
button via plugin parameters or plugin commands!

============================================================================
Features
============================================================================

● Bind common event to keyboard keys or gamepad buttons.
● Remove common events from these keys/buttons.
● Can do it through plugin parameters or plugin commands.

============================================================================
How to use
============================================================================

The plugin is totally self-explanatory, but I want to tell a little tricky 
about it too.

● You can bind some common events to the keys you want through plugin 
parameters. You see that when you go select the key, there is a dropdown 
list for each of them. 
If you don't want to search/scroll the dropdown list, you can use the text 
field to put the key there. It's not case sensitive but has to match the 
keys in the dropdown list.

● Inside game, you can use plugin commands to unbind or bind new common 
events/keys.

The default keys of Game Pad buttons, already assigned by the RM MZ:

"ok", // A
"cancel", // B
"shift", // X
"menu", // Y
"pageup", // LB
"pagedown", // RB
"up", // D-pad up
"down", // D-pad down
"left", // D-pad left
"right" // D-pad right

You can overwrite them changing the plugin parameter.

*NOTE¹: The dropdown list for the gamepad buttons, will have a "g_" prefix
before some of the button names. Don't worry about it. If you see:
"g_a" - Means A button.

etc...

============================================================================
Terms of Use
============================================================================

https://www.hakuenstudio.com/rpg-maker/terms-of-use

============================================================================
Links
============================================================================

Facebook - https://www.facebook.com/hakuenstudio
Instagram - https://www.instagram.com/hakuenstudio
Twitter - https://twitter.com/hakuen_studio

============================================================================
Update log
============================================================================
Version 2.1.1 - 05/03/2021
- Code clean up
- Need Eli Book 3.3.0 now.

Version 2.1.0 - 03/08/2021
- Add support to bind common event to gamepad buttons

Version 2.0.1 - 01/12/2021
- Fixed a bug that was letting you overwrite the default keys even if the 
parameter is set to false.

Version 2.0.0 - 12/18/2020
- Adapted to work with Eli Book 3.0.0.

Version 1.1.0 - 12/06/2020
- Add parameter to choose if you want to overwrite the default keys of 
the RPG Maker.

Version 1.0.1 - 11/29/2020
- Fixed a bug to find the proper keyboard key.

Version 1.0.0 - 11/21/2020
- Plugin release!

@command assign
@text Assign
@desc Assing a common event to a key.

@arg key
@text Keyboard button
@type select
@option a @option b @option c @option d @option e @option f @option g @option h @option i @option j @option k @option l @option m @option n @option o @option p @option q @option r @option s @option t @option u @option v @option w @option x @option y @option z @option 0 @option 1 @option 2 @option 3 @option 4 @option 5 @option 6 @option 7 @option 8 @option 9 @option backspace @option tab @option enter @option shift @option ctrl @option alt @option pausebreak @option capslock @option esc @option space @option pageup @option pagedown @option end @option home @option leftarrow @option uparrow @option rightarrow @option downarrow @option insert @option delete @option leftwindowkey @option rightwindowkey @option selectkey @option numpad0 @option numpad1 @option numpad2 @option numpad3 @option numpad4 @option numpad5 @option numpad6 @option numpad7 @option numpad8 @option numpad9 @option multiply" @option add @option subtract @option decimalpoint @option divide @option f1 @option f2 @option f3 @option f4 @option f5 @option f6 @option f7 @option f8 @option f9 @option f10 @option f11 @option f12 @option numlock @option scrolllock @option semicolon @option equalsign @option comma @option dash @option period @option forwardslash @option graveaccent @option openbracket @option backslash @option closebracket @option singlequote
@desc Type here the keyboard key you want to use. It is not case sensitive.
@default a

@arg commonEvent
@text Common Event Id
@type common_event
@desc The common event Id to assign to that key.
@default 0

@command remove
@text Remove
@desc Remove a common event of a key.

@arg key
@text Keyboard button
@type select
@option a @option b @option c @option d @option e @option f @option g @option h @option i @option j @option k @option l @option m @option n @option o @option p @option q @option r @option s @option t @option u @option v @option w @option x @option y @option z @option 0 @option 1 @option 2 @option 3 @option 4 @option 5 @option 6 @option 7 @option 8 @option 9 @option backspace @option tab @option enter @option shift @option ctrl @option alt @option pausebreak @option capslock @option esc @option space @option pageup @option pagedown @option end @option home @option leftarrow @option uparrow @option rightarrow @option downarrow @option insert @option delete @option leftwindowkey @option rightwindowkey @option selectkey @option numpad0 @option numpad1 @option numpad2 @option numpad3 @option numpad4 @option numpad5 @option numpad6 @option numpad7 @option numpad8 @option numpad9 @option multiply" @option add @option subtract @option decimalpoint @option divide @option f1 @option f2 @option f3 @option f4 @option f5 @option f6 @option f7 @option f8 @option f9 @option f10 @option f11 @option f12 @option numlock @option scrolllock @option semicolon @option equalsign @option comma @option dash @option period @option forwardslash @option graveaccent @option openbracket @option backslash @option closebracket @option singlequote
@desc Choose your key.
@default a

@command assignGamePad
@text Assign (Game Pad)
@desc Assing a common event to a game pad key.

@arg key
@text Game pad button
@type select
@option a @option b @option x @option y @option lb @option rb @option lt @option rt @option select @option start @option l3 @option r3 @option up @option down @option left @option right

@arg commonEvent
@text Common Event Id
@type common_event
@desc The common event Id to assign to that game pad key.
@default 0

@command removeGamePad
@text Remove (Game Pad)
@desc Remove a common event of a key.

@arg key
@text Game pad button
@type select
@option a @option b @option x @option y @option lb @option rb @option lt @option rt @option select @option start @option l3 @option r3 @option up @option down @option left @option right

@param presetKeys
@text Button Common Events
@type struct<presetKeysSt>[]
@desc Set here all your default keys/common events.
@default 

@param presetKeysGamePad
@text Button Common Events(Game Pad)
@type struct<presetGamePadKeysSt>[]
@desc Set here all your default keys/common events.
(Game Pad)
@default 

@param overwrite
@text Overwrite keys
@type boolean
@desc Set to true if you want to overwrite the default keys.
@default false

*/

/*~struct~presetKeysSt:

@param key
@text Keyboard Key
@type select
@option a @option b @option c @option d @option e @option f @option g @option h @option i @option j @option k @option l @option m @option n @option o @option p @option q @option r @option s @option t @option u @option v @option w @option x @option y @option z @option 0 @option 1 @option 2 @option 3 @option 4 @option 5 @option 6 @option 7 @option 8 @option 9 @option backspace @option tab @option enter @option shift @option ctrl @option alt @option pausebreak @option capslock @option esc @option space @option pageup @option pagedown @option end @option home @option leftarrow @option uparrow @option rightarrow @option downarrow @option insert @option delete @option leftwindowkey @option rightwindowkey @option selectkey @option numpad0 @option numpad1 @option numpad2 @option numpad3 @option numpad4 @option numpad5 @option numpad6 @option numpad7 @option numpad8 @option numpad9 @option multiply" @option add @option subtract @option decimalpoint @option divide @option f1 @option f2 @option f3 @option f4 @option f5 @option f6 @option f7 @option f8 @option f9 @option f10 @option f11 @option f12 @option numlock @option scrolllock @option semicolon @option equalsign @option comma @option dash @option period @option forwardslash @option graveaccent @option openbracket @option backslash @option closebracket @option singlequote
@desc The keyboard key. It's not case sensitive.
@default a

@param id
@text Common Event Id
@type common_event
@desc The common event Id.
@default 0

*/

/*~struct~presetGamePadKeysSt:

@param key
@text Game pad button
@type select
@option a @option b @option x @option y @option lb @option rb @option lt @option rt @option select @option start @option l3 @option r3 @option up @option down @option left @option right

@param id
@text Common Event Id
@type common_event
@desc The common event Id.
@default 0

*/

"use strict"

var Eli = Eli || {};
var Imported = Imported || {};
Imported.Eli_ButtonCommonEvents = true;

/* ========================================================================== */
/*                                    ALERT                                   */
/* ========================================================================== */

{

    const installWarning = `You must have installed the EliMZ_Book plugin above all Eli plugins.
Please download it for free.`
    const pluginName = (() => {
        const url = String(document.currentScript._url);
        const start = url.indexOf('Eli');
        const end = url.length - 3;
        const pluginName = url.substring(start, end);

        return pluginName;
    })();
    const requiredVersion = ['3','3','0']
    const updateWarning = `${pluginName} needs an updated version of EliMZ_Book.
Please download it for free.`

    function callEliBook(){
        window.open('https://hakuenstudio.itch.io/')
    };
    
    function needInstallBook() {
        if(!Eli.alert){

            if(window.confirm(installWarning)) callEliBook();
            Eli.alert = true;
        }
    };

    function needUpdateBook() {
        if(!Eli.alert){

            if(window.confirm(updateWarning)) callEliBook();
            Eli.alert = true;
        }
    };
    
    if(!Imported.Eli_Book) needInstallBook();
    if(Eli.Book.Version < requiredVersion) needUpdateBook();
     
}

/* ========================================================================== */
/*                                   PLUGIN                                   */
/* ========================================================================== */

{

Eli.ButtonCommonEvents = {

    parameters: EliPluginManager.createParameters() || {},
    alias: this.alias || {},

    initialize(){
        EliPluginManager.registerCommands(this)
    },

    param(){
        return this.parameters
    },

    getInitialData(isGamePad){
        const initialData = {}

        if(isGamePad){
            for(const data of this.param().presetKeysGamePad){
                const key = String(data.key).toLowerCase()
                initialData[key] = data.id
                this.addToGamePadMapper(data.key)
            }
        }else{
            for(const data of this.param().presetKeys){
                const key = String(data.key).toLowerCase()
                initialData[key] = data.id
                this.addToKeyMapper(data.key)
            }
        }


        return initialData
    },

    data(isGamePad){
        if(isGamePad){
            return $eliData.gamePadCommonEvents()
        }else{
            return $eliData.buttonCommonEvents()
        }
        
    },

    setData(keyName, commonEventId, isGamePad){
        if(isGamePad){
            $eliData.gamePadCommonEvents()[keyName] = commonEventId
        }else{
            $eliData.buttonCommonEvents()[keyName] = commonEventId
        }
        
    },

    isValidKey(keyName){
        return $eliData.buttonCommonEvents().hasOwnProperty(keyName)
    },

    assign(args){
        const keyName = args.key.toLowerCase()
        const commonEvent = args.commonEvent
        this.addToKeyMapper(keyName)
        this.setData(keyName, +commonEvent)
    },

    remove(args){
        const keyName = args.key.toLowerCase()
        const keyCode = Input.keyBoardCodes[keyName]

        delete Input.keyMapper[keyCode]
        delete this.data()[keyName]
    },

    assignGamePad(args){
        const keyName = args.key.toLowerCase()
        const commonEvent = args.commonEvent

        this.addToGamePadMapper(keyName)
        this.setData(keyName, +commonEvent, true)
    },

    removeGamePad(args){
        const keyName = args.key.toLowerCase()
        const keyCode = Input.gamePadCodes[keyName]
        const isGamePad = true

        delete Input.gamepadMapper[keyCode]
        delete this.data(isGamePad)[keyName]
    },

    addToKeyMapper(keyName){
        const keyValue = keyName.toLowerCase()
        const keyCode = Input.keyBoardCodes[keyValue]

        if(!this.isDefaultKey(keyValue)){
            Input.keyMapper[keyCode] = keyValue

        }else if(this.parameters.overwrite){
            Input.keyMapper[keyCode] = keyValue
        }
    },

    addToGamePadMapper(keyName){
        const keyValue = keyName.toLowerCase()
        const keyCode = Input.gamePadCodes[keyValue]

        if(!this.isDefaultGamePadKey(keyValue)){
            Input.gamepadMapper[keyCode] = keyValue

        }else if(this.parameters.overwrite){
            Input.gamepadMapper[keyCode] = keyValue
        }
    },

    isDefaultKey(keyName){
        const defaultKeys = [
            "tab", "ok", "shift", "control", "escape", "ok", "pageup", "pagedown", 
            "left", "up", "right", "down", "debug"
        ]

        return defaultKeys.includes(keyName)
    },

    isDefaultGamePadKey(keyName){
        const defaultKeys = [
            "ok", "cancel", "shift", "menu", "ok", "pageup", "pagedown", 
            "left", "up", "right", "down"
        ]

        return defaultKeys.includes(keyName)
    },

    loadKeyMapperChanges(){
        for(let keyName in this.data()){
            Plugin.addToKeyMapper(keyName)
        }
        for(let keyName in this.data(true)){
            Plugin.addToGamePadMapper(keyName)
        }
    },

    updateGamePad(){
        const index = Input._gamepadStates[0].indexOf(true)

        if(index !== -1){
            const key = Input.gamepadMapper[index]
            const commonEventId = this.data(true)[key]

            if(commonEventId){
                $gameTemp.reserveCommonEvent(commonEventId)
                Input._gamepadStates[0].fill(false)
            }

        }
    },

    updateKeyboard(){
        const commonEventId = this.data()[Input._latestButton]
        $gameTemp.reserveCommonEvent(commonEventId)
        Input._latestButton = null
    },

    update(){
        if(!$gameMap.isEventRunning()){
            if(this.isValidKey(Input._latestButton)){
                this.updateKeyboard()
            }
            if(Input._gamepadStates.length > 0){
                this.updateGamePad()
            }
        }

        
    },

}

const Alias = Eli.ButtonCommonEvents.alias
const Plugin = Eli.ButtonCommonEvents

Plugin.initialize()

/* ========================================================================== */
/*                                  SAVE DATA                                 */
/* ========================================================================== */

Alias.Eli_SavedContents_initialize = Eli_SavedContents.prototype.initialize
Eli_SavedContents.prototype.initialize = function(){
    Alias.Eli_SavedContents_initialize.call(this)
    this.contents.buttonCommonEvents = Plugin.getInitialData()
    this.contents.gamePadCommonEvents = Plugin.getInitialData(true)
}

Eli_SavedContents.prototype.buttonCommonEvents = function(){
    return this.contents.buttonCommonEvents
}

Eli_SavedContents.prototype.gamePadCommonEvents = function(){
    return this.contents.gamePadCommonEvents
}

/* ========================================================================== */
/*                                   MANAGER                                  */
/* ========================================================================== */

Alias.DataManager_extractSaveContents = DataManager.extractSaveContents
DataManager.extractSaveContents = function(contents) {
    Alias.DataManager_extractSaveContents.call(this, contents)
    Plugin.loadKeyMapperChanges()
}

/* ========================================================================== */
/*                                    SCENE                                   */
/* ========================================================================== */

Alias.Scene_Map_afterUpdate = Scene_Map.prototype.afterUpdate
Scene_Map.prototype.afterUpdate = function(){
    Alias.Scene_Map_afterUpdate.call(this)
    Plugin.update()
}

}